#ifndef __POLYGON3D__
#define __POLYGON3D__

#include <mem.h>
#include "Vector3D.h"
#include "Matrix3D.h"
#include "Plane.h"
#include "Texture.h"

#define  IN_POSITIVE  0
#define  IN_NEGATIVE  1
#define  IN_BOTH      2
#define  IN_PLANE     3

class    Polygon3D
{
public:
    Plane    * plane;       // plane through this facet
    int        numVertices; // current # of vertices
    int        maxVertices; // storage allocated for maxVertices
    Vector3D * vertices;    // polygon vertices
    Vector3D * uvMap;       // texture coordinates (u,v,1)
    Texture  * texture;     // texture of this polygon

    Polygon3D ()
    {
        plane       = NULL;
        vertices    = NULL;
        uvMap       = NULL;
        texture     = NULL;
        numVertices = 0;
        maxVertices = 0;
    }

    Polygon3D ( const Polygon3D& poly )
    {
        plane       = new Plane ( *poly.getPlane () );
        numVertices = poly.numVertices;
        maxVertices = poly.maxVertices;
        vertices    = new Vector3D [maxVertices];

        memcpy ( vertices, poly.vertices, sizeof (Vector3D) * numVertices );

        if ( poly.uvMap != NULL )
        {
            uvMap = new Vector3D [maxVertices];
            memcpy ( uvMap, poly.uvMap, sizeof (Vector3D) * numVertices );
        }
        else
            uvMap = NULL;

        texture  = (Texture *)  poly.texture -> retain ();
    }

    Polygon3D ( int n , int allocUVMap )
    {
        numVertices = 0;
        maxVertices = ((n + 7) >> 3 ) << 3;
        vertices    = new Vector3D [maxVertices];

        if ( allocUVMap )
            uvMap = new Vector3D [maxVertices];
        else
            uvMap = NULL;

        texture  = NULL;
    }

    Polygon3D ( int n, Vector3D * v, Vector3D * uv = NULL )
    {
        numVertices = n;
        maxVertices = ((n + 7) >> 3 ) << 3;
        vertices    = new Vector3D [maxVertices];

        memcpy ( vertices, v, sizeof (Vector3D) * numVertices );

        if ( uv != NULL )
        {
            uvMap = new Vector3D [maxVertices];
            memcpy ( uvMap, uv, sizeof (Vector3D) * numVertices );
        }
        else
            uvMap = NULL;

        texture  = NULL;

        computePlane ();
    }

    ~Polygon3D ()
    {
        if ( vertices != NULL )
           delete vertices;

        if ( uvMap != NULL )
           delete uvMap;

        if ( plane != NULL )
           delete plane;

        if ( texture != NULL )
           texture -> release ();
    }

    Polygon3D& operator = ( const Polygon3D& poly );

    Plane * getPlane () const
    {
        return plane;
    }

    int  isEmpty () const
    {
        return numVertices < 1;
    }

    int  isFrontFacing ( const Vector3D& viewDir ) const
    {
        return plane != NULL ? ( plane -> n & viewDir ) < 0 : 0;
    }

    void    setTexture ( Texture * tex )
    {
        if ( texture != NULL )
            texture -> release ();

        texture = (Texture *) tex -> retain ();
    }

    void        transform ( const Matrix3D& );      // apply affine transform
    void        translate ( const Vector3D& );      // translate polygon
    int         classify  ( const Plane& p ) const;
    Polygon3D * split     ( const Plane& p );        // return part in negative halfspace
    void        addVertex ( const Vector3D& v );
    void        addVertex ( const Vector3D& v, const Vector3D& uv );
    void        delVertex ( int index );

protected:
    void        realloc      ( int newMaxVertices );
    void        computePlane ();
};

#endif
