#include    <assert.h>
#include    "Poly3d.h"

#define EPS         1e-7
#define MAX_POINTS  15

Polygon3D& Polygon3D :: operator = ( const Polygon3D& poly )
{
    if ( plane != NULL )
       delete plane;

    if ( vertices != NULL )
       delete vertices;

    if ( uvMap != NULL )
       delete uvMap;

    if ( texture != NULL )
       texture -> release ();

    plane       = new Plane ( *poly.getPlane () );
    numVertices = poly.numVertices;
    maxVertices = poly.maxVertices;
    vertices    = new Vector3D [maxVertices];

    memcpy ( vertices, poly.vertices, sizeof (Vector3D) * numVertices );

    if ( poly.uvMap != NULL )
    {
        uvMap = new Vector3D [maxVertices];
        memcpy ( uvMap, poly.uvMap, sizeof (Vector3D) * numVertices );
    }
    else
        uvMap = NULL;

    setTexture  ( poly.texture  );

    return *this;
}

void    Polygon3D :: transform ( const Matrix3D& tr )
{
    for ( register int i = 0; i < numVertices; i++ )
        vertices [i] = tr * vertices [i];
}

void    Polygon3D :: translate ( const Vector3D& v )
{
    for ( register int i = 0; i < numVertices; i++ )
        vertices [i] += v;
}

int	    Polygon3D :: classify ( const Plane& p ) const
{
	int	positive = 0;
	int	negative = 0;

	for ( int i = 0; i < numVertices; i++ )
	{
		float	res = p.f ( vertices [i] );

		if ( res > EPS )
			positive++;
		else
		if ( res < -EPS )
			negative++;
	}

	if ( positive > 0 && negative == 0 )
		return IN_POSITIVE;
	else
	if ( positive == 0 && negative > 0 )
		return IN_NEGATIVE;
	else
	if ( positive < 1 && negative < 1 )
		return IN_PLANE;
	else
		return IN_BOTH;
}

Polygon3D * Polygon3D :: split ( const Plane& p )
{
	Vector3D p1   [MAX_POINTS];
	Vector3D p2   [MAX_POINTS];
    Vector3D p1uv [MAX_POINTS];
    Vector3D p2uv [MAX_POINTS];
	Vector3D prevP  = vertices [numVertices - 1];
    Vector3D prevUV;
	float	 prevF  = p.f ( prevP );
	int	     count1 = 0;
	int	     count2 = 0;

    if ( uvMap != NULL )
       prevUV = uvMap [numVertices-1];

	for ( int i = 0; i < numVertices; i++ )
	{
		Vector3D curP = vertices [i];
        Vector3D curUV;
		float	 curF = p.f ( curP );

        if ( uvMap != NULL )
           curUV = uvMap [i];

		if ( curF >= 0 && prevF <= 0 )
        {
			p1   [count1]   = curP;
            p1uv [count1++] = curUV;
        }
		else
		if ( curF < 0 && prevF >= 0 )
        {
			p2   [count2]   = curP;
            p2uv [count2++] = curUV;
        }
		else
		if ( curF < 0 && prevF > 0 )
		{
			float	 t    = - curF / ( prevF - curF );
			Vector3D sp   =  curP + t * ( prevP - curP );

            if ( uvMap != NULL )
            {
                  Vector3D spUV = curUV + t * ( prevUV - curUV );

                  p1uv [count1]   = spUV;
                  p2uv [count2]   = spUV;
                  p2uv [count2+1] = curUV;
            }

			p1 [count1++] = sp;
			p2 [count2++] = sp;
			p2 [count2++] = curP;
		}
		else
		if ( curF > 0 && prevF < 0 )
		{
			float	 t  = - curF / ( prevF - curF );
			Vector3D sp =  curP + t * ( prevP - curP );

            if ( uvMap != NULL )
            {
                  Vector3D spUV = curUV + t * ( prevUV - curUV );

                  p1uv [count1]   = spUV;
                  p2uv [count2]   = spUV;
                  p1uv [count1+1] = curUV;
            }

			p1 [count1++] = sp;
			p2 [count2++] = sp;
			p1 [count1++] = curP;
		}

	  	prevP  = curP;
		prevF  = curF;
        prevUV = curUV;
	}

    if ( ( numVertices = count1 ) >= maxVertices )
       realloc ( numVertices );

    memcpy ( vertices, p1, sizeof (Vector3D) * numVertices );

    if ( uvMap != NULL )
       memcpy ( uvMap, p1uv, sizeof (Vector3D) * numVertices );

    return new Polygon3D ( count2, p2, uvMap != NULL ? p2uv : NULL );
}

void    Polygon3D :: addVertex ( const Vector3D& v )
{
    assert ( uvMap == NULL );

    if ( numVertices + 1 >= maxVertices )
       realloc ( maxVertices + 8 );

    vertices [numVertices++] = v;
}

void    Polygon3D :: addVertex ( const Vector3D& v, const Vector3D& uv )
{
    assert ( ( uvMap != NULL ) || ( vertices == NULL && uvMap == NULL ) );

    if ( numVertices + 1 >= maxVertices )
       realloc ( maxVertices + 8 );

    vertices [numVertices]   = v;
    uvMap    [numVertices++] = uv;
}

void    Polygon3D :: delVertex ( int index )
{
    if ( isEmpty () )
       return;

    if ( index < numVertices - 1 )
    {
        memmove ( &vertices [index], &vertices [index+1],
                  sizeof (Vector3D) * (numVertices - index - 1) );

        if ( uvMap != NULL )
            memmove ( &uvMap [index], &uvMap [index+1],
                 sizeof (Vector3D) * (numVertices - index - 1) );
    }

    numVertices--;
}

void    Polygon3D :: realloc ( int newMaxVertices )
{
    maxVertices = ( (newMaxVertices + 7) >> 3 ) << 3;

    Vector3D * tmp;

    tmp      = vertices;
    vertices = new Vector3D [maxVertices];

    memcpy ( vertices, tmp, sizeof (Vector3D) * numVertices );

    tmp   = uvMap;
    uvMap = new Vector3D [maxVertices];

    memcpy ( uvMap, tmp, sizeof (Vector3D) * numVertices );
}

void    Polygon3D :: computePlane ()
{
    assert ( numVertices > 2 );

    Vector3D n ( (vertices [1] - vertices [0]) ^ (vertices [2] - vertices [1]) );

    n.normalize ();

    float    d = - (vertices [0] & n );

    plane = new Plane ( n, d );
}
