#ifndef __POLYGON2D__
#define __POLYGON2D__

#include <memory.h>
#include "Pool.h"
#include "BBox.h"
#include "Texture.h"
#include "Vector3D.h"

enum Orientation
{
    POSITIVE,
    NEGATIVE
};

class Polygon2D
{
public:
    int          numVertices;
    int          maxVertices;
    Vector2D   * vertices;
    Vector3D   * uvMap;
    BoundingBox  boundingBox;
    Texture    * texture;
    Orientation  orientation;

    Polygon2D ()
    {
        numVertices = 0;
        maxVertices = 0;
        vertices    = NULL;
        uvMap       = NULL;
        texture     = NULL;
        orientation = POSITIVE;
    }

    Polygon2D ( int n , int allocUVMap )
    {
        numVertices = 0;
        maxVertices = ((n + 7) >> 3 ) << 3;
        vertices    = (Vector2D *) pool -> alloc (sizeof (Vector2D) * maxVertices);
        orientation = POSITIVE;

        if ( allocUVMap )
            uvMap = (Vector3D *) pool -> alloc (sizeof (Vector3D) * maxVertices);
        else
            uvMap = NULL;

        texture  = NULL;
    }

    Polygon2D ( int n, Vector2D * v, Vector3D * uv )
    {
        numVertices = n;
        maxVertices = ((n + 7) >> 3 ) << 3;
        vertices    = (Vector2D *) pool -> alloc (sizeof (Vector2D) * maxVertices);
        orientation = POSITIVE;

        memcpy ( vertices, v, sizeof (Vector2D) * numVertices );

        if ( uv != NULL )
        {
            uvMap = (Vector3D *) pool -> alloc (sizeof (Vector3D) * maxVertices);
            memcpy ( uvMap, uv, sizeof (Vector3D) * numVertices );
        }
        else
            uvMap = NULL;

        texture  = NULL;

        for ( int i = 0; i < numVertices; i++ )
            boundingBox.addVertex ( vertices [i] );
    }

    Polygon2D ( const Polygon2D& poly )
    {
        numVertices = poly.numVertices;
        maxVertices = poly.maxVertices;
        vertices    = (Vector2D *) pool -> alloc (sizeof (Vector2D) * maxVertices);
        orientation = poly.orientation;

        memcpy ( vertices, poly.vertices, sizeof (Vector2D) * numVertices );

        if ( poly.uvMap != NULL )
        {
            uvMap = (Vector3D *) pool -> alloc (sizeof (Vector3D) * maxVertices);
            memcpy ( uvMap, poly.uvMap, sizeof (Vector3D) * numVertices );
        }
        else
            uvMap = NULL;

        texture     = (Texture *)  poly.texture  -> retain ();
        boundingBox = poly.boundingBox;
    }

    ~Polygon2D ()
    {
        if ( vertices != NULL )
           pool -> free ( vertices );

        if ( uvMap != NULL )
           pool -> free ( uvMap );

        if ( texture != NULL )
           texture -> release ();
    }

    void * operator new ( size_t size )
    {
        return pool -> alloc ( size );
    }

    void operator delete ( void * ptr )
    {
        pool -> free ( ptr );
    }

    Polygon2D& operator = ( const Polygon2D& poly );

    int        operator == ( const Polygon2D& poly ) const;

    Vector2D& operator [] ( int i )
    {
        return vertices [i];
    }

    int isEmpty () const
    {
        return numVertices < 1;
    }

    void    setTexture ( Texture * tex )
    {
        if ( texture != NULL )
            texture -> release ();

        texture = (Texture *) tex -> retain ();
    }

    void addVertex ( const Vector2D& v );
    void addVertex ( const Vector2D& v, const Vector3D& uv );
    void delVertex ( int vertexIndex );
    void intersect ( const Polygon2D& clipPoly );

protected:
    void  realloc ( int newMaxVertices );
};

#endif
