#include "Poly2D.h"

#define MAX_VERTICES  100

int Polygon2D :: operator == ( const Polygon2D& poly ) const
{
    if ( numVertices != poly.numVertices )
       return 0;

    if ( orientation != poly.orientation )
        return 0;

    for( register int i = 0; i < numVertices; i++ )
        if ( fastDist ( vertices [i], poly.vertices [i] ) > EPS )
           return 0;

    if ( uvMap != NULL )
    {
        if ( poly.uvMap == NULL )
           return 0;

        for ( register int i = 0; i < numVertices; i++ )
            if ( fabs(uvMap[i].x-poly.uvMap[i].x)>EPS || fabs(uvMap[i].y-poly.uvMap[i].y)>EPS )
               return 0;
    }

    return -1;
}

Polygon2D& Polygon2D :: operator = ( const Polygon2D& poly )
{
    pool -> free ( vertices );

    if( uvMap != NULL)
        pool -> free ( uvMap );

    numVertices = poly.numVertices;
    maxVertices = poly.maxVertices;
    boundingBox = poly.boundingBox;
    orientation = poly.orientation;

    if ( maxVertices < 1 )
    {
        vertices = NULL;
        uvMap    = NULL;
    }
    else
    {
        vertices = (Vector2D *)(pool -> alloc (sizeof (Vector2D) * maxVertices));

        memcpy ( vertices, poly.vertices, sizeof (Vector2D) * numVertices );

        if( poly.uvMap != NULL )
        {
            uvMap = (Vector3D *) pool -> alloc (sizeof(Vector3D) * maxVertices);

            memcpy ( uvMap, poly.uvMap, sizeof (Vector3D) * numVertices );
        }
        else
            uvMap = NULL;

        setTexture  ( poly.texture );
    }

    return *this;
}

void Polygon2D :: addVertex ( const Vector2D& v )
{
    assert ((texture==NULL) && (uvMap==NULL));

    if ( numVertices + 1 >= maxVertices )
        realloc ( numVertices + 1 );

    vertices[numVertices++] = v;

    boundingBox.addVertex ( v );
}

void Polygon2D :: addVertex (const Vector2D& v, const Vector3D& uv )
{
    assert((texture!=NULL) && (uvMap!=NULL));

    if ( numVertices + 1 >= maxVertices )
        realloc ( numVertices + 1 );

    vertices [numVertices]   = v;
    uvMap    [numVertices++] = uv;

    boundingBox.addVertex ( v );
}

void  Polygon2D :: delVertex ( int index )
{
    if ( isEmpty () )
       return;

    assert ( index < numVertices );

    if ( index < numVertices - 1 )
    {
        memmove ( &vertices [index], &vertices [index+1],
                  sizeof (Vector2D) * (numVertices - index - 1) );

        if ( uvMap != NULL )
            memmove ( &uvMap [index], &uvMap [index+1],
                 sizeof (Vector3D) * (numVertices - index - 1) );
    }

    numVertices--;

    boundingBox.reset ();

    for ( register int i = 0; i < numVertices; i++ )
        boundingBox.addVertex ( vertices [i] );
}

// clip polygon to a halfplane a*x + b*y + c >= 0

inline  void    clipPolyByPlane ( Vector2D * inPoly,  Vector3D * inPolyMap,
                                  Vector2D * outPoly, Vector3D * outPolyMap,
                                  int& inCount, int& outCount,
                                  float a, float b, float c )
{
    int     prevIndex = inCount - 1;
    float   prevF     = a*inPoly [prevIndex].x + b*inPoly [prevIndex].y + c;
    int     i = 0;

    outCount = 0;

    while ( i < inCount )
    {
        float   curF = a*inPoly [i].x + b*inPoly [i].y + c;

        if ( curF >= 0.0 && prevF < 0.0 )
        {
            float   t = curF / (curF - prevF);

            if ( t > EPS && t < 1.0 - EPS )
            {
                outPoly [outCount] = inPoly [i] + t*(inPoly [prevIndex] - inPoly [i]);

                outPolyMap [outCount++] = inPolyMap [i] +
                    t*(inPolyMap [prevIndex] - inPolyMap [i]);
            }

            outPoly    [outCount]   = inPoly [i];
            outPolyMap [outCount++] = inPolyMap [i];
        }
        else
        if ( curF >= 0.0 && prevF >= 0.0 )
        {
            outPoly    [outCount]   = inPoly [i];
            outPolyMap [outCount++] = inPolyMap [i];
        }
        else
        if ( curF < 0.0 && prevF >= 0.0 )
        {
            float   t = curF / (curF - prevF);

            if ( t > EPS && t < 1.0 - EPS )
            {
                outPoly [outCount] = inPoly [i] +
                    t*(inPoly [prevIndex] - inPoly [i]);

                outPolyMap [outCount++] = inPolyMap [i] +
                    t*(inPolyMap [prevIndex] - inPolyMap [i]);
            }
        }

        prevF     = curF;
        prevIndex = i++;
    }
}

// clip using Sutherland-Hodgeman clipping method

void Polygon2D :: intersect ( const Polygon2D& clipPoly )
{
    Vector2D   outPoly1    [MAX_VERTICES];
    Vector2D   outPoly2    [MAX_VERTICES];
    Vector3D   outPoly1Map [MAX_VERTICES];
    Vector3D   outPoly2Map [MAX_VERTICES];
    Vector2D * inPoly      = vertices;
    Vector2D * outPoly     = outPoly1;
    Vector3D * inPolyMap   = (uvMap != NULL ? uvMap : outPoly2Map);
    Vector3D * outPolyMap  = outPoly1Map;
    int        outVertices = 0;
    int        inVertices  = numVertices;

    if( !boundingBox.overlap (clipPoly.boundingBox) )   // no intersection
    {
        numVertices = 0;

        return;
    }

    int prev = clipPoly.numVertices - 1;                // prevoius vertex index
    int cur = 0;                                        // current vertex index

    for ( ; cur < clipPoly.numVertices; prev = cur++ )  // clip by every edge [clipPoly.vertices [prev], clipPoly.vertices [cur]]
    {                                                   // compute line coeffs.
        float   a = clipPoly.vertices [cur ].y - clipPoly.vertices [prev].y;
        float   b = clipPoly.vertices [prev].x - clipPoly.vertices [cur ].x;
        float   c = -a*clipPoly.vertices [cur].x - b*clipPoly.vertices [cur].y;

        if ( clipPoly.orientation != POSITIVE )
        {
            a = -a;
            b = -b;
            c = -c;
        }
                                                        // clip poly by current line
        clipPolyByPlane ( inPoly, inPolyMap, outPoly, outPolyMap,
                          inVertices, outVertices, a, b, c );

        if ( outVertices < 3 )                          // empty
            break;

        inPoly     = outPoly;
        inPolyMap  = outPolyMap;
        inVertices = outVertices;

        if ( inPoly == outPoly1 )
        {
            outPoly    = outPoly2;
            outPolyMap = outPoly2Map;
        }
        else
        {
            outPoly    = outPoly1;
            outPolyMap = outPoly1Map;
        }
    }

    boundingBox.reset ();

    if ( outVertices < 3 )
    {
        numVertices = 0;

        return;
    }

    if ( (numVertices = outVertices) >= maxVertices )
        realloc ( numVertices );

    memcpy ( vertices, inPoly, numVertices * sizeof ( Vector2D ) );

    if ( uvMap != NULL )
        memcpy ( uvMap,    inPolyMap, numVertices * sizeof ( Vector3D ) );

    for ( register int i = 0; i < numVertices; i++ )
        boundingBox.addVertex ( vertices [i] );
}

void    Polygon2D :: realloc ( int newMaxVertices )
{
    maxVertices = ((newMaxVertices + 7)>>3)<<3;

    Vector2D * tmp1 = vertices;
    Vector3D * tmp2 = uvMap;

    vertices = (Vector2D*) pool -> alloc ( sizeof(Vector2D) * maxVertices );

    memcpy ( vertices, tmp1, sizeof (Vector2D) * numVertices );

    pool -> free ( tmp1 );

    if ( tmp2 != NULL )   // uvMap present
    {
        uvMap = (Vector3D *) pool -> alloc ( sizeof (Vector3D) * maxVertices );

        memcpy ( uvMap, tmp2, sizeof (Vector3D) * numVertices );

        pool -> free ( tmp2 );
    }
}
