#include    "Poly2D.h"
#include    "FixMath.h"
#include    "Point.h"

#define     MAX_VERTICES    20

int         xStep     = 16;         // step for partially linear mapping
int         xStepBits = 4;          // xStep == (1 << xStepBits)

                                    // draw mapped line using partialy-linear
                                    // approximation
inline void mapPartiallyLinearLine ( const Vector3D& map1, const Vector3D& map2,
                                     int x1, int x2, int y,
                                     Texture * texture, char far * videoMem )
{
    if ( x2 < x1 )
       return;
                                    // prepare texture params
    int    widthPower = texture -> widthPower;
    int    widthMask  = texture -> width - 1;
    int    heightMask = texture -> height - 1;
    char * data = texture -> data;

    videoMem += x1;               // adjust video memory pointer

    if ( x2 == x1 )               // degenerate segment
    {
        int u = map1.x / map1.z;
        int v = map1.y / map1.z;

        * videoMem = data [u + (v << widthPower)];

        return;
    }
                                    // setup partially-linear approximation
    Vector3D    dMap ((map2 - map1) * (xStep / (float)(x2 - x1 + 1)));
    Vector3D    map (map1);
    int         numSegments = (x2 - x1 + 1) >> xStepBits;
    Fixed       u1          = float2Fixed ( map1.x / map1.z + 0.5f );
    Fixed       v1          = float2Fixed ( map1.y / map1.z + 0.5f );
    Fixed       u           = u1;
    Fixed       v           = v1;
    Fixed       u2, v2;
    Fixed       du;
    Fixed       dv;
                                // draw segment by chunks of xStep width
    for ( int i = 0; i < numSegments; i++ )
    {
        map += dMap;
        u2   = float2Fixed ( map.x / map.z + 0.5f );
        v2   = float2Fixed ( map.y / map.z + 0.5f );
        du   = ( u2 - u1 ) >> xStepBits;
        dv   = ( v2 - v1 ) >> xStepBits;

                                // now draw the chunk
        for ( register int x = 0; x < xStep; x++ )
        {
            * videoMem++ = data [(widthMask   & fixed2Int(u)) +
                                 ((heightMask & fixed2Int(v)) << widthPower)];
            u           += du;
            v           += dv;
        }

        u1  = u2;
        v1  = v2;
        x1 += xStep;
    }

                                // now draw the rest of line
    if ( x1 > x2 )
       return;

    u2 = float2Fixed ( map2.x / map2.z + 0.5f );
    v2 = float2Fixed ( map2.y / map2.z + 0.5f );
    du = (u2 - u1)/(x2 - x1+1);
    dv = (v2 - v1)/(x2 - x1+1);

    for ( ; x1 <= x2; x1++ )
    {
       * videoMem++ = data [(widthMask   & fixed2Int(u)) +
                            ((heightMask & fixed2Int(v)) << widthPower)];
       u           += du;
       v           += dv;
    }
}

static	int	findEdge ( int& i, int dir, Point * p, int numVertices )
{
	for ( ; ; )
	{
		int	i1 = i + dir;

		if ( i1 < 0 )
			i1 = numVertices - 1;
		else
		if ( i1 >= numVertices )
			i1 = 0;

		if ( p [i1].y < p [i].y )	// edge [i,i1] is going upwards
			return -1;  		// must be some error
		else
		if ( p [i1].y == p [i].y )	// horizontal edge
			i = i1;
		else				// edge [i, i1] is going downwords
			return i1;
	}
}

void    drawPartiallyLinearMappedPoly ( const Polygon2D& poly, char far * videoMem )
{
    Point      p [MAX_VERTICES];
    int        numVertices = poly.numVertices;
    Vector2D * vertices    = poly.vertices;
    Vector3D * uvMap       = poly.uvMap;

    for ( int i = 0; i < numVertices; i++ )
    {
        p [i].x = (int) ( vertices [i].x + 0.5f );
        p [i].y = (int) ( vertices [i].y + 0.5f );
    }

    int yMin          = p [0].y;
    int yMax          = yMin;
    int topPointIndex = 0;

                                    // find y-range and topPointIndex
    for ( i = 1; i < numVertices; i++ )
        if ( p [i].y < p [topPointIndex].y )
            topPointIndex = i;
            else
            if ( p [i].y > yMax )
                yMax = p [i].y;

    yMin = p [topPointIndex].y;

    if ( yMin >= yMax )         // degenerate poly
        return;

    int   i1      = topPointIndex;
    int   i2      = topPointIndex;
    int   i1Next  = findEdge  ( i1, -1, p, numVertices );
    int   i2Next  = findEdge  ( i2, 1, p, numVertices );
    Fixed x1      = int2Fixed ( p [i1].x );
    Fixed x2      = int2Fixed ( p [i2].x );
	Fixed dx1     = fraction2Fixed (p [i1Next].x - p [i1].x, p [i1Next].y - p [i1].y);
	Fixed dx2     = fraction2Fixed (p [i2Next].x - p [i2].x, p [i2Next].y - p [i2].y);

    Vector3D map1  ( uvMap [i1] );
    Vector3D map2  ( uvMap [i2] );
    Vector3D dMap1 ((uvMap [i1Next] - uvMap [i1]) / (p [i1Next].y - p [i1].y));
    Vector3D dMap2 ((uvMap [i2Next] - uvMap [i2]) / (p [i2Next].y - p [i2].y));

                                // now process each span in turn
    for ( int y = yMin; y <= yMax; y++ )
    {
                                // draw the span
        mapPartiallyLinearLine ( map1, map2, fixed2Int (x1), fixed2Int (x2), y, poly.texture, videoMem + 320*y );

                                // adjust vars to next line
        map1 += dMap1;
        map2 += dMap2;
        x1   += dx1;
        x2   += dx2;

                                // check for coming through vertex
		if ( y + 1 >= p [i1Next].y )
		{
			i1 = i1Next;		// switch to next edge
			if ( --i1Next < 0 )
				i1Next = numVertices - 1;
					     		// check for lower horizontal part
			if ( p [i1].y >= p [i1Next].y )
				break;

                                // adjust deltas for x and uv
			dx1   = fraction2Fixed (p [i1Next].x - p [i1].x, p [i1Next].y - p [i1].y);
            dMap1 = (uvMap [i1Next] - uvMap [i1]) / (p [i1Next].y - p [i1].y);
            x1    = int2Fixed (p [i1].x);
            map1  = uvMap [i1];
		}

		if ( y + 1 >= p [i2Next].y )
		{
			i2 = i2Next;	    // switch to next edge
			if ( ++i2Next >= numVertices )
				i2Next = 0;
							    // check for lower horizontal part
			if ( p [i2].y >= p [i2Next].y )
				break;

                                // adjust deltas for x and uv
			dx2   = fraction2Fixed (p [i2Next].x - p [i2].x, p [i2Next].y - p [i2].y);
            dMap2 = (uvMap [i2Next] - uvMap [i2]) / (p [i2Next].y - p [i2].y);
            x2    = int2Fixed ( p [i2].x );
            map2  = uvMap [i2];
		}
    }
}

