//
// Camera class.
// Makes the transform x'=transf*(x-pos) from world to camera space
// Method project does perspective projection of 3D vector onto the
// screen.
//
// Author: Alex V. Boreskoff
// Last change 20/10/2000
//

#ifndef __CAMERA__
#define __CAMERA__

#include    "3DDefs.h"
#include    "Matrix3D.h"
#include    "Plane.h"
#include    "Object.h"

class   Camera : public Object
{
public:
    Vector3D pos;           // camera position
    Vector3D viewDir;       // viewing direction
    Vector3D upDir;         // up direction
    Vector3D rightDir;
    Matrix3D transf;        // camera transform (from world to camera space)
    float    xScale;
    float    yScale;
    float    fov;           // field of view angle
    Plane    clippingPlane;
    int      mirrored;      // whether the camera is mirrored

    Camera ( char * theName, const Vector3D& p, const Vector3D& lookAt, const Vector3D& up,
            float anXScale, float anYScale, float aFov );

    Camera ( const Camera& camera );

    virtual char * getClassName () const
    {
        return "Camera";
    }

    void    rotate        ( float yaw, float pitch, float roll );
    void    setFov        ( float newFovAngle );
    void    computeMatrix ();
    void    mirror        ( const Plane& );

    int     getCullingCodes ( const Vector3D& v ) const;

    int     inViewingFrustrum ( const Vector3D& v ) const
    {
        return getCullingCodes ( v ) == 0;
    }

    void    move  ( const Vector3D& newPos )
    {
        pos = newPos;
    }

    void    transform ( const Vector3D& p, Vector3D& res ) const
    {
        res = transf * ( p - pos );
    }

    void    setClippingPlane ( const Plane& plane )
    {
        clippingPlane = plane;
    }

    int isMirrored () const
    {
        return mirrored;
    }

    float   getFov () const
    {
        return fov;
    }

    Plane getClippingPlane () const
    {
        return clippingPlane;
    }

    float   angle2Fov ( float angle ) const
    {
        return 0.5f / tan ( 0.5f * M_PI * angle / 180.0f );
    }
};

#endif
