//
// Simple camera class.
// Makes the transform x'=transf*(x-pos) from world to camera space
// Method project does perspective projection of 3D vector onto the
// screen.
//
// Author: Alex V. Boreskoff
// Last change 18/01/2001
//

#include    "Camera.h"

Camera :: Camera ( char * theName, const Vector3D& p, const Vector3D& lookAt, const Vector3D& up,
            float anXScale, float anYScale, float aFov ) : Object ( theName ), clippingPlane ( lookAt - p, EPS )
{
    xScale   = anXScale;
    yScale   = anYScale;
    fov      = aFov;
    pos      = p;
    viewDir  = lookAt - pos;
    upDir    = up;
    mirrored = 0;

    computeMatrix ();
}

Camera :: Camera ( const Camera& camera ) : Object ( camera.getName () ), clippingPlane ( camera.clippingPlane )
{
    xScale   = camera.xScale;
    yScale   = camera.yScale;
    fov      = camera.fov;
    pos      = camera.pos;
    viewDir  = camera.viewDir;
    upDir    = camera.upDir;
    mirrored = camera.mirrored;

    computeMatrix ();
}

void    Camera :: rotate ( float yaw, float pitch, float roll )
{
    Matrix3D rot ( rotateYPR ( yaw, pitch, roll ) );

    viewDir  = rot * viewDir;
    upDir    = rot * upDir;
    rightDir = rot * rightDir;

    clippingPlane.rotate ( rot );

    computeMatrix ();
}

void    Camera :: setFov ( float newFovAngle )
{
    fov = newFovAngle;

    computeMatrix ();
}

void    Camera :: computeMatrix ()
{
   viewDir.normalize ();        // orthonormalize viewDir
                                // compute upDir
   upDir   -= (upDir & viewDir) * viewDir;

                                // compute rightDir
   rightDir = upDir ^ viewDir;

   upDir.normalize   ();        // orthonormalize upDir and rightDir
   rightDir.normalize ();

   float  fovValue = angle2Fov ( fov );

   transf [0][0] = rightDir [0] * xScale * fovValue;
   transf [0][1] = rightDir [1] * xScale * fovValue;
   transf [0][2] = rightDir [2] * xScale * fovValue + 0.5f * xScale;
   transf [1][0] = -upDir   [0] * yScale * fovValue;
   transf [1][1] = -upDir   [1] * yScale * fovValue;
   transf [1][2] = -upDir   [2] * yScale * fovValue + 0.5f * yScale;
   transf [2][0] = viewDir  [0];
   transf [2][1] = viewDir  [1];
   transf [2][2] = viewDir  [2];
}

void    Camera :: mirror ( const Plane& mirror )
{
    mirror.mirrorPos   ( pos );
    mirror.mirrorDir   ( viewDir );
    mirror.mirrorDir   ( upDir );
    mirror.mirrorDir   ( rightDir );
    mirror.mirrorPlane ( clippingPlane );

    mirrored = !mirrored;

    computeMatrix ();
}

int Camera :: getCullingCodes ( const Vector3D& v ) const
{
    Vector3D    p   = transf*(v - pos);
    int         res = 0;

    if ( p.x < 0.0f )           // left plane
       res |= 0x01;

    if ( p.x > p.z * xScale )   // right plane
       res |= 0x02;

    if ( p.y < 0.0f )           // top plane
       res |= 0x04;

    if ( p.y > p.z * yScale )   // bottom plane
       res |= 0x08;

    if ( p.z > EPS )            // near plane
       res |= 0x10;

    return res;
}
