// StarSystem.cpp
//

#include "Master.h"
#include "StarSystem.h"

bool CStarSystem::Init(CInfo &info)
{
	char szKey[_MAX_PATH];
	CStar::Init(info);
	m_nPlanets = info.GetIntValue("Planets");
	if(!m_nPlanets)
		return false;
	m_pPlanets = new CPlanet[m_nPlanets];
	for(int i=0; i<m_nPlanets; i++)
	{
		sprintf(szKey, "%s.%d", m_szName, i+1);
		m_pPlanets[i].SetParent(this);
		m_pPlanets[i].Init(CInfo(info, szKey));
	}
	return true;
}

void CStarSystem::Cleanup()
{
	if(m_pPlanets)
	{
		delete[] m_pPlanets;
		m_pPlanets = NULL;
	}
}

void CStarSystem::Update(C3DObject *pCamera, float fSeconds, int nPolygonMode, float fMaxError, bool bUpdate, bool bTexture)
{
	m_llPlanets.RemoveAll();

	// Handle collision detection and distance-based sorting
	CPlanet *pCollision = NULL;
	for(int i=0; i<GetPlanetCount(); i++)
	{
		CPlanet *pPlanet = GetPlanet(i);
		for(int j=0; j<=pPlanet->GetMoonCount(); j++)
		{
			CPlanet *pCurrent = (j < pPlanet->GetMoonCount()) ? pPlanet->GetMoon(j) : pPlanet;
			if(pCurrent->CollisionCheck(pCamera))
				pCollision = pCurrent;
			for(CPtrListNode *pNode=m_llPlanets.GetHead(); pNode->IsInList(); pNode = pNode->GetNext())
			{
				if(((CPlanet *)pNode->GetData())->GetHeight() > pCurrent->GetHeight())
					break;
			}
			pNode->InsertBefore(pCurrent);
		}
	}

	if(pCollision)
	{
		// Very cheesy collision response for now
		pCamera->SetVelocity(-pCamera->GetVelocity());
		CDoubleVector v = pCamera->GetPosition() - pCollision->GetPosition();
		double dScale = (DELTA - pCollision->GetHeight()) / v.Magnitude();
		v *= dScale;
		pCamera->SetPosition(pCamera->GetPosition() + v);
	}

	for(CPtrListNode *pNode=m_llPlanets.GetHead(); pNode->IsInList(); pNode = pNode->GetNext())
	{
		CPlanet *pPlanet = (CPlanet *)pNode->GetData();
		if(pPlanet->GetImpostorError(pCamera) >= 0.00002f)
		{
			if(pPlanet->GetImpostorScreenSpace(pCamera) > 0.75f)
			{
				pPlanet->ClearFlags(CImpostor::Enabled | CImpostor::NeedsUpdate);
				pPlanet->SetViewpoint(pCamera, 0);
			}
			else
			{
				pPlanet->SetFlags(CImpostor::Enabled | CImpostor::NeedsUpdate);
				CMatrix mView;
				pPlanet->GetImpostorViewMatrix(pCamera, mView);
				mView.Transpose();
				C3DObject objView(mView);
				objView.SetPosition(pCamera->GetPosition());
				pPlanet->SetViewpoint(&objView, 0);
			}
			if(bUpdate)
				pPlanet->Update(fMaxError);
		}
	}
}

void CStarSystem::Draw(C3DObject *pCamera, float fSeconds, int nPolygonMode, float fMaxError, bool bUpdate, bool bTexture)
{
	CPtrListNode *pNode;
	CPlanet *pPlanet;
	GLUtil()->BeginVertexArray();

	// Start by updating any impostors that need to be updated
	//GLUtil()->BeginPBuffer();
	glPolygonMode(GL_FRONT, nPolygonMode);
	for(pNode=GetHeadPlanet(); pNode->IsInList(); pNode = pNode->GetNext())
	{
		pPlanet = (CPlanet *)pNode->GetData();
		if(pPlanet->GetFlags(CImpostor::NeedsUpdate) != 0)
		{
			pPlanet->InitImpostorRender(pCamera);
			glPolygonMode(GL_FRONT, nPolygonMode);
			glClearColor(0.05f, 0.05f, 0.05f, 0.0f);
			glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
			pPlanet->Draw(pCamera, this, bTexture, true);
			pPlanet->FinishImpostorRender();
		}
	}
	glPolygonMode(GL_FRONT, GL_FILL);
	//GLUtil()->EndPBuffer();

	glClearColor(0.0f, 0.0f, 0.0f, 0.0f);
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	// Draw the billboards in the star system
	glDisable(GL_LIGHTING);
	glDisable(GL_DEPTH_TEST);
	CStar::Draw(pCamera);
	for(pNode = GetTailPlanet(); pNode->IsInList(); pNode = pNode->GetPrevious())
	{
		pPlanet = (CPlanet *)pNode->GetData();
		if(pPlanet->GetFlags(CImpostor::Enabled))
			pPlanet->DrawImpostor(pCamera);
	}
	glEnable(GL_DEPTH_TEST);
	glEnable(GL_LIGHTING);

	// If the nearest planet is being rendered normally, render it here
	pPlanet = (CPlanet *)GetHeadPlanet()->GetData();
	if(pPlanet->GetFlags(CImpostor::Enabled) == 0)
	{
		glPolygonMode(GL_FRONT, nPolygonMode);
		pPlanet->Draw(pCamera, this, bTexture, false);
		glPolygonMode(GL_FRONT, GL_FILL);
	}
	GLUtil()->EndVertexArray();
}
