// Planet.h
//

#include "PlanetInfo.h"
#include "Impostor.h"
#include "ROAMSphere.h"
#include "Star.h"

class CPlanet : public CImpostor, public CROAMSphere
{
protected:
	CHeightMap m_mapHeight;
	CSurfaceMap m_mapSurface;

	CVector m_vCameraPosition;			// The position of the camera (relative to the planet)
	CQuaternion m_qCameraOrientation;	// The orientation of the camera (relative to the planet)
	float m_fHeight;					// The distance between the camera and the ground directly beneath it
	float m_fAltitude;					// The distance between the camera and the planet (using average radius)
	float m_fHorizon;					// The distance from the camera to the planet's horizon (squared)

	// Members loaded from config file
	char m_szName[_MAX_NAME+1];
	float m_fOrbitalDistance;		// The planet's average orbital distance from its star
	float m_fAngularVelocity;		// The speed at which the planet rotates (radians/sec)

	int m_nColors;
	CColor *m_pColors;
	int m_nMoons;
	CPlanet *m_pMoons;

public:
	CPlanet()
	{
		m_pColors = NULL;
		m_pMoons = NULL;
	}
	~CPlanet()
	{
		Cleanup();
	}
	void Init(CInfo &info);
	void Cleanup();

	const char *GetName()	{ return m_szName; }
	int GetMoonCount()		{ return m_nMoons; }
	CPlanet *GetMoon(int i)	{ return &m_pMoons[i]; }

	float GetHeight()		{ return m_fHeight; }
	float GetAltitude()		{ return m_fAltitude; }
	float GetHorizon()		{ return m_fHorizon; }
	float GetOrbitalDistance()	{ return m_fOrbitalDistance; }

	void SetViewpoint(C3DObject *pCamera, float fSeconds)
	{
		Rotate(CVector(0, 1, 0), m_fAngularVelocity*fSeconds);
		m_vCameraPosition = UnitInverse().RotateVector(pCamera->GetPosition() - GetPosition());
		m_qCameraOrientation = *pCamera;
		m_qCameraOrientation.Rotate(UnitInverse());

		float fOriginalHeight = m_vCameraPosition.Magnitude();
		m_fHeight = fOriginalHeight - m_mapHeight.GetHeight(m_vCameraPosition, fOriginalHeight);
		m_fAltitude = Max(1.0f, fOriginalHeight - m_mapHeight.GetRadius());
		m_fHorizon = m_fAltitude*m_fAltitude + 2.0f*m_fAltitude*m_mapHeight.GetRadius();	// Distance to horizon squared
	}
	bool CollisionCheck()	{ return m_fHeight < DELTA; }
	bool CollisionCheck(C3DObject *pCamera)
	{
		CVector vCameraPosition = UnitInverse().RotateVector(pCamera->GetPosition() - GetPosition());
		float fOriginalHeight = vCameraPosition.Magnitude();
		m_fHeight = fOriginalHeight - m_mapHeight.GetHeight(m_vCameraPosition, fOriginalHeight);
		return m_fHeight < DELTA;
	}

	void Update(float fMaxError)
	{
		CVector vHeading = m_qCameraOrientation.GetViewAxis();
		CROAMSphere::Update(m_vCameraPosition, vHeading, m_fHorizon, fMaxError);
	}
	void Draw(C3DObject *pCamera, CStar *pStar, bool bTexture=true, bool bImpostor=false);
};
