// Planet.cpp
//

#include "Master.h"
#include "Planet.h"


void CPlanet::Init(CInfo &info)
{
	Cleanup();

	// Load some values into local variables for initializing the maps
	int nSeed = info.GetIntValue("Seed");
	int nResolution = info.GetIntValue("Resolution");
	float fRadius = info.GetFloatValue("Radius");
	float fMaxHeight = info.GetFloatValue("MaxHeight");
	float fWaterLevel = info.GetFloatValue("WaterLevel");
	float fOctaves = info.GetFloatValue("Octaves");
	float fRoughness = info.GetFloatValue("Roughness");

	// Load some values into permanent members
	strcpy(m_szName, info.GetTextValue("Name"));
	m_fMass = info.GetFloatValue("Mass");
	m_fOrbitalDistance = info.GetFloatValue("OrbitalDistance") * 0.1f;
	m_fAngularVelocity = info.GetFloatValue("AngularVelocity");
	m_nMoons = info.GetIntValue("Moons");
	m_nColors = info.GetIntValue("Colors");

	// Load the color map
	if(m_nColors)
	{
		m_pColors = new CColor[m_nColors];
		for(int i=0; i<m_nColors; i++)
		{
			char szName[20];
			sprintf(szName, "Color%d", i+1);
			m_pColors[i] = info.GetColorValue(szName);
		}
	}

	// Initialize the planetary maps and ROAM spheres
	m_mapHeight.InitFractal(nSeed, fOctaves, fRoughness);
	m_mapHeight.Init(0, fRadius, fMaxHeight, fWaterLevel);
	m_mapSurface.Init(0, &m_mapHeight, m_pColors, m_nColors);
	CROAMSphere::Init(&m_mapHeight, &m_mapSurface);

	// Initialize impostor values
	m_fBoundingRadius = fRadius * 1.1f;
	CDoubleVector vPosition = m_pParent->GetPosition();
	vPosition.x += m_fOrbitalDistance;
	SetPosition(vPosition);

	// Last but not least, set up any moons that might revolve around this planet
	if(m_nMoons)
	{
		m_pMoons = new CPlanet[m_nMoons];
		char szKey[_MAX_PATH];
		for(int i=0; i<m_nMoons; i++)
		{
			sprintf(szKey, "%s.%d", m_szName, i+1);
			m_pMoons[i].SetParent(this);
			m_pMoons[i].Init(CInfo(info, szKey));
		}
	}
}

void CPlanet::Cleanup()
{
	if(m_pMoons)
	{
		delete[] m_pMoons;
		m_pMoons = NULL;
	}
	if(m_pColors)
	{
		delete[] m_pColors;
		m_pColors = NULL;
	}
}

void CPlanet::Draw(C3DObject *pCamera, CStar *pStar, bool bTexture, bool bImpostor)
{
	// First set up the lighting for this planet
	float fAmbient = 0.05f;
	CVector vLight1 = pStar->GetPosition()-GetPosition();
	glEnable(GL_LIGHT0);
	glLightfv(GL_LIGHT0, GL_DIFFUSE, CVector4(1.5f));
	glLightfv(GL_LIGHT0, GL_POSITION, CVector4(vLight1.x, vLight1.y, vLight1.z, 1));

	// Multiply the current view matrix by the model matrix and draw the planet
	CMatrix m;
	float fScale = 1.0f;

	if(!bImpostor)
	{
		fScale = GetScaledModelMatrix(m, pCamera);
		glPushMatrix();
		glMultMatrixf(m);
	}
	
	glEnable(GL_NORMALIZE);		// We have to enable this because we're using a scaled MODELVIEW matrix

	float fBlendRatio = (m_fAltitude / m_mapHeight.GetRadius() - 0.25f);
	bool bFaceTextures = false;
	bool bCrossFade = false;
	if(bTexture)
	{
		bFaceTextures =  m_mapSurface.GetResolution() != 0 && (fBlendRatio > 0.0f);
		if(bFaceTextures)
		{
			// We're far enough away that it's better to use the pre-computed face textures
			GLUtil()->EnableTextureCoordArray(GL_TEXTURE0_ARB, 2);
			bCrossFade = (fBlendRatio < 1.0f) && GLUtil()->GetMaxTextureUnits() > 1 && GLUtil()->HasRegisterCombiners();
			if(bCrossFade)
			{
				// Gradually cross-fade between face textures and raw texture
				glActiveTextureARB(GL_TEXTURE1_ARB);
				GLUtil()->EnableTextureCoordArray(GL_TEXTURE1_ARB, 0);
				m_mapSurface.GetRawTextureMap().Enable();
				glActiveTextureARB(GL_TEXTURE0_ARB);

				glEnable(GL_REGISTER_COMBINERS_NV);
				fBlendRatio = CMath::fastsqrt(fBlendRatio);
				glCombinerParameterfvNV(GL_CONSTANT_COLOR0_NV, CVector4(fBlendRatio, fBlendRatio, fBlendRatio, fBlendRatio));
				glCombinerInputNV(GL_COMBINER0_NV, GL_RGB, GL_VARIABLE_A_NV, GL_PRIMARY_COLOR_NV, GL_UNSIGNED_IDENTITY_NV, GL_RGB);
				glCombinerInputNV(GL_COMBINER0_NV, GL_RGB, GL_VARIABLE_B_NV, GL_TEXTURE0_ARB, GL_UNSIGNED_IDENTITY_NV, GL_RGB);
				glCombinerInputNV(GL_COMBINER0_NV, GL_RGB, GL_VARIABLE_C_NV, GL_PRIMARY_COLOR_NV, GL_UNSIGNED_IDENTITY_NV, GL_RGB);
				glCombinerInputNV(GL_COMBINER0_NV, GL_RGB, GL_VARIABLE_D_NV, GL_TEXTURE1_ARB, GL_UNSIGNED_IDENTITY_NV, GL_RGB);
				glCombinerOutputNV(GL_COMBINER0_NV, GL_RGB, GL_SPARE0_NV, GL_SPARE1_NV, GL_DISCARD_NV, GL_NONE, GL_NONE, 0, 0, 0);
				glFinalCombinerInputNV(GL_VARIABLE_A_NV, GL_CONSTANT_COLOR0_NV, GL_UNSIGNED_IDENTITY_NV, GL_RGB);
				glFinalCombinerInputNV(GL_VARIABLE_A_NV, GL_CONSTANT_COLOR0_NV, GL_UNSIGNED_IDENTITY_NV, GL_ALPHA);
				glFinalCombinerInputNV(GL_VARIABLE_B_NV, GL_SPARE0_NV, GL_UNSIGNED_IDENTITY_NV, GL_RGB);
				glFinalCombinerInputNV(GL_VARIABLE_C_NV, GL_SPARE1_NV, GL_UNSIGNED_IDENTITY_NV, GL_RGB);
				/* The code above should be the same as...
				char szBuffer[8192];
				sprintf(szBuffer, 
					"!!RC1.0\n"
					"const0 = (%f, %f, %f, %f);\n"
					"{\n"
					"  rgb {\n"
					"    spare0 = col0 * tex0;\n"
					"    spare1 = col0 * tex1;\n"
					"  }\n"
					"}\n"
					"out.rgb = lerp(const0, spare0, spare1);\n"
					"out.a = unsigned_invert(zero);\n",
					fRatio, fRatio, fRatio, fRatio);
				nvparse(szBuffer);
				*/
			}
		}
		else
		{
			// We're close enough to use the raw texture and detailed noise texture
			GLUtil()->EnableTextureCoordArray(GL_TEXTURE0_ARB, 0);
			m_mapSurface.GetRawTextureMap().Enable();
			glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
			if(GLUtil()->GetMaxTextureUnits() > 1)
			{
				glActiveTextureARB(GL_TEXTURE1_ARB);
				GLUtil()->EnableTextureCoordArray(GL_TEXTURE1_ARB, 2);
				CTexture::GetNoise().Enable();
				glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
				glMatrixMode(GL_TEXTURE);
				glPushMatrix();
				glLoadIdentity();
				glScalef(128, 128, 128);
				glActiveTextureARB(GL_TEXTURE0_ARB);
			}
		}
	}

	BuildIndexList();
	for(int nFace=0; nFace<6; nFace++)
	{
		if(bFaceTextures)
			m_mapSurface.GetTextureMap(nFace).Enable();
		DrawFace(nFace);
		if(bFaceTextures)
			m_mapSurface.GetTextureMap(nFace).Disable();
	}

	if(bTexture)
	{
		if(bCrossFade)
			glDisable(GL_REGISTER_COMBINERS_NV);

		if(GLUtil()->GetMaxTextureUnits() > 1)
		{
			glActiveTextureARB(GL_TEXTURE1_ARB);
			GLUtil()->DisableTextureCoordArray(GL_TEXTURE1_ARB);
			CTexture::Disable(GL_TEXTURE_1D);	// Needed for now because raw texture is 1D. This should change later.
			CTexture::Disable(GL_TEXTURE_2D);
			if(!bFaceTextures)
			{
				glPopMatrix();
				glMatrixMode(GL_MODELVIEW);
			}
			glActiveTextureARB(GL_TEXTURE0_ARB);
		}
		GLUtil()->DisableTextureCoordArray(GL_TEXTURE0_ARB);
		CTexture::Disable(GL_TEXTURE_1D);	// Needed for now because raw texture is 1D. This should change later.
		CTexture::Disable(GL_TEXTURE_2D);
	}

	glDisable(GL_NORMALIZE);	// We have to enable this because we're using a scaled MODELVIEW matrix
	if(!bImpostor)
		glPopMatrix();
}
