// Impostor.h
//

#ifndef __Impostor_h__
#define __Impostor_h__

#include "Texture.h"

class CImpostor : public C3DObject
{
protected:
	CVector m_vOrientation;		// The orientation of the camera relative to the impostor at the time it was rendered
	float m_fDistance;			// The distance to the camera relative to the impostor at the time it was rendered
	CVector m_vUp;				// The up vector used to generate the impostor texture

	short m_nFlags;				// A set of bit flags for keeping track of various states
	short m_nResolution;		// The chosen resolution for the billboard texture map
	float m_fBillboardRadius;	// The billboard radius (drawn as a square, so it's the same as half the length of one edge)
	CTexture m_tBillboard;		// The billboard texture map

	// Static members to hold state between calls to InitImpostorRender() and FinishImpostorRender()
	static int m_nViewport[4];

public:
	enum { NoFlags = 0x0000, Enabled = 0x0001, NeedsUpdate = 0x0002, NewTexture = 0x0004, AllFlags = 0xFFFF };

	CImpostor() : C3DObject()
	{
		m_nResolution = 0;
		m_nFlags = NoFlags;
	}

	void InitImpostorRender(C3DObject *pCamera);
	void FinishImpostorRender();
	void DrawImpostor(C3DObject *pCamera);

	short GetFlags(short nFlags)	{ return m_nFlags & nFlags; } 
	void SetFlags(short nFlags)		{ m_nFlags |= nFlags; } 
	void ClearFlags(short nFlags)	{ m_nFlags &= ~nFlags; } 

	float GetImpostorError(C3DObject *pCamera)
	{
		float fError = 1.0f;
		if(GetFlags(NeedsUpdate) != 0 || GetFlags(Enabled) == 0)
			return fError;
		CVector vView = m_vPosition - pCamera->GetPosition();
		float fDistance = vView.Magnitude();
		CVector vOrientation = UnitInverse().RotateVector(-vView);
		float fDelta = 0.01f * Abs(m_fDistance-fDistance) / m_fDistance;
		float fAngle = 1 - Abs((m_vOrientation | vOrientation) / fDistance);
		fError = Max(fAngle, fDelta);
		return fError;
	}
	void GetImpostorViewMatrix(CMatrix &mView, CVector &vView)
	{
		CVector vAt(0.0f);
		CVector vUp = CVector(vView.y, -vView.z, vView.x);
		mView.ViewMatrix(vAt, vView, vUp);
	}
	void GetImpostorViewMatrix(C3DObject *pCamera, CMatrix &mView)
	{
		CVector vView = m_vPosition - pCamera->GetPosition();
		GetImpostorViewMatrix(mView, vView);
	}

	float GetImpostorScreenSpace(float fDistance)
	{
		float fAltitude = Max(fDistance - m_fBoundingRadius, DELTA);				// Height above the surface of the bounding radius
		float fHorizon = CMath::fastsqrt(fAltitude*fAltitude + 2.0f*fAltitude*m_fBoundingRadius);	// Distance to horizon (use pythagorean theorem to get the equation)
		float fCos = fHorizon / fDistance;											// Cosine of angle between ray to center of sphere and ray to horizon (part of a right triangle)
		float fTemp2 = fDistance / fCos;
		float fBillboardRadius = CMath::fastsqrt(fTemp2*fTemp2 - fDistance*fDistance);	// Distance from center to sides at the center of the object (size of the billboard)
		return fBillboardRadius / fDistance;
	}
	float GetImpostorScreenSpace(C3DObject *pCamera)
	{
		return GetImpostorScreenSpace(DistanceTo(*pCamera));
	}
	short GetImpostorResolution(float fScreenSpace)
	{
		short nResolution;
		if(fScreenSpace > 1.0f)
			nResolution = 0;
		else if(fScreenSpace > 0.5f)
			nResolution = 512;
		else if(fScreenSpace > 0.2f)
			nResolution = 256;
		else if(fScreenSpace > 0.08f)
			nResolution = 128;
		else if(fScreenSpace > 0.032f)
			nResolution = 64;
		else if(fScreenSpace > 0.0128f)
			nResolution = 32;
		else if(fScreenSpace > 0.00512f)
			nResolution = 16;
		else if(fScreenSpace > 0.002048f)
			nResolution = 8;
		else
			nResolution = 4;
		return nResolution;
	}
	short GetImpostorResolution(C3DObject *pCamera)
	{
		return GetImpostorResolution(GetImpostorScreenSpace(pCamera));
	}
};

#endif // __Impostor_h__
