// Impostor.cpp
//

#include "Master.h"
#include "Impostor.h"
#include "GLUtil.h"

int CImpostor::m_nViewport[4];

void CImpostor::InitImpostorRender(C3DObject *pCamera)
{
	// Calculate and store the camera orientation and distance for calculating future errors
	CVector vView = m_vPosition - pCamera->GetPosition();
	m_vOrientation = UnitInverse().RotateVector(-vView);
	m_fDistance = m_vOrientation.Magnitude();
	m_vOrientation /= m_fDistance;

	// Get the suggested resolution for this impostor and see if it's changed since last time
	short nOldResolution = m_nResolution;
	m_nResolution = GetImpostorResolution(GetImpostorScreenSpace(m_fDistance));
	if(nOldResolution != m_nResolution)
		SetFlags(NewTexture);
	GLUtil()->BeginPBuffer(m_nResolution);

	// Calculate the model-view matrix for rendering the impostor, and get the up vector for rendering the billboard
	CMatrix mModel, mView;
	mModel.ModelMatrix(*this, vView);
	GetImpostorViewMatrix(mView, vView);
	m_vUp = CVector(mView.f12, mView.f22, mView.f32);

	// Set up the necessary viewport, modelview matrix, and projection matrix for rendering to the impostor texture
	glGetIntegerv(GL_VIEWPORT, m_nViewport);
	glViewport(0, 0, m_nResolution, m_nResolution);
	glMatrixMode(GL_PROJECTION);
	glPushMatrix();
	glLoadIdentity();

	// Time for some fancy geometry
	float fAltitude = m_fDistance - m_fBoundingRadius;							// Height above the surface of the bounding radius
	float fHorizon = CMath::fastsqrt(fAltitude*fAltitude + 2.0f*fAltitude*m_fBoundingRadius);	// Distance to horizon (use pythagorean theorem to get the equation)
	float fCos = fHorizon / m_fDistance;										// Cosine of angle between ray to center of sphere and ray to horizon (part of a right triangle)
	float fTemp = fAltitude / fCos;												// Horizon distance cut off at the near clipping plane
	float fBound = CMath::fastsqrt(fTemp*fTemp - fAltitude*fAltitude);			// Distance from center to sides at the near clipping plane (boundaries of the projection frustum) (part of another right triangle)
	float fTemp2 = m_fDistance / fCos;
	m_fBillboardRadius = CMath::fastsqrt(fTemp2*fTemp2 - m_fDistance*m_fDistance);	// Distance from center to sides at the center of the object (size of the billboard)
	glFrustum(-fBound, fBound, -fBound, fBound, m_fDistance-m_fBoundingRadius, m_fDistance+m_fBoundingRadius);

	glMatrixMode(GL_MODELVIEW);
	glPushMatrix();
	glLoadMatrixf(mView);
	glPushMatrix();
	glMultMatrixf(mModel);
}

void CImpostor::FinishImpostorRender()
{
	// Copy the currently rendered viewport into the impostor texture
	if(GetFlags(NewTexture) != 0)
		m_tBillboard.InitCopy(0, 0, m_nResolution, m_nResolution);
	else
		m_tBillboard.UpdateCopy(0, 0, m_nResolution, m_nResolution);
	ClearFlags(NeedsUpdate | NewTexture);

	// Clean up the OpenGL matrix stack and viewport
	glViewport(m_nViewport[0], m_nViewport[1], m_nViewport[2], m_nViewport[3]);
	glMatrixMode(GL_PROJECTION);
	glPopMatrix();
	glMatrixMode(GL_MODELVIEW);
	glPopMatrix();
	glPopMatrix();

	GLUtil()->EndPBuffer();
}

void CImpostor::DrawImpostor(C3DObject *pCamera)
{
	if(GetFlags(Enabled) == 0)
		return;

	// First determine the actual normal, up, and right vectors for the billboard
	CVector vPos = m_vPosition - pCamera->GetPosition();
	float fDistance = vPos.Magnitude();

	CVector vNormal = vPos / -fDistance;
	CVector vRight = m_vUp ^ vNormal;
	vRight.Normalize();
	CVector vUp = vNormal ^ vRight;
	vUp.Normalize();

	float fScale = GetScalingFactor(fDistance);
	vPos *= fScale;
	vRight *= fScale;
	vUp *= fScale;

	// Then draw the billboarded rectangle
	m_tBillboard.Enable();
	glBlendFunc(GL_ONE, GL_ONE_MINUS_SRC_ALPHA);
	glEnable(GL_BLEND);
	glBegin(GL_QUADS);
	glTexCoord2f(0, 1);
	glVertex3fv(vPos + vUp*m_fBillboardRadius + vRight*(-m_fBillboardRadius));
	glTexCoord2f(0, 0);
	glVertex3fv(vPos + vUp*(-m_fBillboardRadius) + vRight*(-m_fBillboardRadius));
	glTexCoord2f(1, 0);
	glVertex3fv(vPos + vUp*(-m_fBillboardRadius) + vRight*m_fBillboardRadius);
	glTexCoord2f(1, 1);
	glVertex3fv(vPos + vUp*m_fBillboardRadius + vRight*m_fBillboardRadius);
	glEnd();
	glDisable(GL_BLEND);
	m_tBillboard.Disable();
}
