/*
	The Universe Development Kit
	Copyright (C) 2000  Sean O'Neil
	s_p_oneil@hotmail.com

	This library is free software; you can redistribute it and/or
	modify it under the terms of the GNU Lesser General Public
	License as published by the Free Software Foundation; either
	version 2.1 of the License, or (at your option) any later version.

	This library is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
	Lesser General Public License for more details.

	You should have received a copy of the GNU Lesser General Public
	License along with this library; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "Master.h"
#include "GameEngine.h"

#include "PlanetInfo.h"

CGameEngine::CGameEngine()
{
	// Initialize tweakable settings
	m_fMaxError = 0.0005f;
	m_nPolygonMode = GL_FILL;
	m_bInfo = true;
	m_bUpdate = true;
	m_bTexture = true;

	// Initialize static textures and the vertex array
	CMath::InitStaticMembers();

	// Initialize the star system and camera objects
	char szPath[_MAX_PATH];
	sprintf(szPath, "%s\\ss.cfg", GetApp()->GetStartupPath());
	CInfo info(szPath, "StarSystem");
	if(!m_starSystem.Init(info))
	{
		MessageBox(NULL, "Unable to initialize solar system. Make sure a valid SS.CFG is in your startup path.", "Initialization Error", MB_OK);
		exit(0);	// This really sucks as far as cleanup is concerned, but at least it keeps it from crashing
	}

	GLUtil()->Init();
	CTexture::InitStaticMembers(76234, 128);

	// Load and initialize the font object
	m_fFont.Init(GLUtil()->GetHDC());

	CDoubleVector vPos = m_starSystem.GetPlanet(2)->GetPosition();
	vPos.z += 20000;
	m_3DCamera.SetPosition(vPos);
	//m_3DCamera.SetPosition(CDoubleVector(0.0, 0.0, 100.0f));

	m_starSystem.Update(&m_3DCamera, 0, m_nPolygonMode, m_fMaxError, m_bUpdate, m_bTexture);
	for(CPtrListNode *pNode=m_starSystem.GetHeadPlanet(); pNode->IsInList(); pNode = pNode->GetNext())
		((CPlanet *)pNode->GetData())->SetFlags(CImpostor::NeedsUpdate);
}

CGameEngine::~CGameEngine()
{
	GLUtil()->Cleanup();
}

void CGameEngine::RenderFrame(int nMilliseconds)
{
	// Determine the FPS
	static char szFrameCount[20] = {0};
	static int nTime = 0;
	static int nFrames = 0;
	nTime += nMilliseconds;
	if(nTime >= 1000)
	{
		float fFPS = (float)(nFrames * 1000) / (float)nTime;
		sprintf(szFrameCount, "%2.2f FPS", fFPS);
		nTime = nFrames = 0;
	}
	nFrames++;

	// Update the star system (movement, collision detection, ROAM and impostor updates)
	float fSeconds = nMilliseconds * 0.001f;
	HandleInput(fSeconds);
	m_starSystem.Update(&m_3DCamera, fSeconds, m_nPolygonMode, m_fMaxError, m_bUpdate, m_bTexture);

	glPushMatrix();
	glLoadMatrixf(m_3DCamera.GetViewMatrix());
	m_starSystem.Draw(&m_3DCamera, fSeconds, m_nPolygonMode, m_fMaxError, m_bUpdate, m_bTexture);
	glPopMatrix();

	// Draw the FPS in the top-left corner
	m_fFont.Begin();
	glColor3d(1.0, 1.0, 1.0);
	m_fFont.SetPosition(10, 460);
	m_fFont.Print(szFrameCount);
	
	if(m_bInfo)
	{
		char szBuffer[_MAX_PATH];
		m_fFont.SetPosition(10, 445);
		sprintf(szBuffer, "Err: %f, Vert: %d", m_fMaxError, CVertex::Array.GetLockedElementCount());
		m_fFont.Print(szBuffer);
		m_fFont.SetPosition(10, 430);
		sprintf(szBuffer, "Speed: %2.2f km/sec", m_3DCamera.GetVelocity().Magnitude());
		m_fFont.Print(szBuffer);
		m_fFont.SetPosition(10, 415);
		CPlanet *pPlanet = (CPlanet *)m_starSystem.GetHeadPlanet()->GetData();
		sprintf(szBuffer, "%s: %2.2f km, Tri: %d", pPlanet->GetName(), (float)pPlanet->GetAltitude(), pPlanet->GetTriangleCount());
		m_fFont.Print(szBuffer);
	}
	
	m_fFont.End();
	glFlush();
}

void CGameEngine::HandleInput(float fSeconds)
{
	for(int i=1; i<=m_starSystem.GetPlanetCount(); i++)
	{
		if(GetKeyState('0'+i) & 0x8000)
		{
			CPlanet *pPlanet = m_starSystem.GetPlanet(i-1);
			CDoubleVector v = pPlanet->GetPosition();
			v.z += 20000;
			m_3DCamera.SetPosition(v);
			m_3DCamera.SetVelocity(CVector(0.0f, 0.0f, 0.0f));
		}
	}

	if(GetKeyState(VK_ADD) & 0x8000)
		m_fMaxError *= 0.98f;
	if(GetKeyState(VK_SUBTRACT) & 0x8000)
		m_fMaxError *= 1.02f;

	// Turn left/right means rotate around the up axis
	if(GetKeyState(VK_NUMPAD6) & 0x8000)
		m_3DCamera.Rotate(m_3DCamera.GetUpAxis(), fSeconds * -0.5f);
	if(GetKeyState(VK_NUMPAD4) & 0x8000)
		m_3DCamera.Rotate(m_3DCamera.GetUpAxis(), fSeconds * 0.5f);

	// Turn up/down means rotate around the right axis
	if(GetKeyState(VK_NUMPAD8) & 0x8000)
		m_3DCamera.Rotate(m_3DCamera.GetRightAxis(), fSeconds * -0.5f);
	if(GetKeyState(VK_NUMPAD2) & 0x8000)
		m_3DCamera.Rotate(m_3DCamera.GetRightAxis(), fSeconds * 0.5f);

	// Roll means rotate around the view axis
	if(GetKeyState(VK_NUMPAD7) & 0x8000)
		m_3DCamera.Rotate(m_3DCamera.GetViewAxis(), fSeconds * -0.5f);
	if(GetKeyState(VK_NUMPAD9) & 0x8000)
		m_3DCamera.Rotate(m_3DCamera.GetViewAxis(), fSeconds * 0.5f);

#define THRUST		100.0f	// Acceleration rate due to thrusters (km/s*s)
#define RESISTANCE	0.00f	// Damping effect on velocity

	// Handle acceleration keys
	CVector vAccel;
	if(GetKeyState(VK_SPACE) & 0x8000)
		m_3DCamera.SetVelocity(CVector(0.0f));	// Full stop
	else
	{
		// Calculate camera's acceleration due to gravity
		//vAccel = m_starSystem.GetPlanet(2)->GravityVector(m_3DCamera);
		vAccel = CVector(0.0f);

		// Add camera's acceleration due to thrusters
		float fThrust = THRUST;
		if(GetKeyState(VK_CONTROL) & 0x8000)
			fThrust *= 1000.0f;

		// Thrust forward/reverse affects velocity along the view axis
		if(GetKeyState('W') & 0x8000)
			vAccel += m_3DCamera.GetViewAxis() * fThrust;
		if(GetKeyState('S') & 0x8000)
			vAccel += m_3DCamera.GetViewAxis() * -fThrust;

		// Thrust left/right affects velocity along the right axis
		if(GetKeyState('D') & 0x8000)
			vAccel += m_3DCamera.GetRightAxis() * fThrust;
		if(GetKeyState('A') & 0x8000)
			vAccel += m_3DCamera.GetRightAxis() * -fThrust;

		m_3DCamera.Accelerate(vAccel, fSeconds, RESISTANCE);
	}

	// Handle moon's acceleration due to gravity
	//vAccel = m_plEarth.GravityVector(m_rsMoon);
	//m_rsMoon.Accelerate(vAccel, fSeconds, RESISTANCE);
}
