/*
	The Universe Development Kit
	Copyright (C) 2000  Sean O'Neil
	s_p_oneil@hotmail.com

	This library is free software; you can redistribute it and/or
	modify it under the terms of the GNU Lesser General Public
	License as published by the Free Software Foundation; either
	version 2.1 of the License, or (at your option) any later version.

	This library is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
	Lesser General Public License for more details.

	You should have received a copy of the GNU Lesser General Public
	License along with this library; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "Master.h"
#include "GameApp.h"


CWinApp *CWinApp::m_pMainApp;

int WINAPI WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, char *pszCmdLine, int nShowCmd)
{
	CGameApp app(hInstance, hPrevInstance, pszCmdLine, nShowCmd);
	if(app.InitInstance())
		app.Run();
	return app.ExitInstance();
}

void CDisplayMode::LoadDisplaySettings()
{
	m_bFullScreen = (GetApp()->GetProfileInt(DISPLAY_SECTION, "FullScreen", 1) != 0);
	m_nWidth = GetApp()->GetProfileInt(DISPLAY_SECTION, "Width", 640);
	m_nHeight = GetApp()->GetProfileInt(DISPLAY_SECTION, "Height", 480);
	m_nBitsPerPixel = m_bFullScreen ? GetApp()->GetProfileInt(DISPLAY_SECTION, "BitsPerPixel", 16) : 0;
	m_nFrequency = m_bFullScreen ? GetApp()->GetProfileInt(DISPLAY_SECTION, "Frequency", 0) : 0;
}

void CDisplayMode::SaveDisplaySettings()
{
	GetApp()->WriteProfileInt(DISPLAY_SECTION, "FullScreen", m_bFullScreen);
	GetApp()->WriteProfileInt(DISPLAY_SECTION, "Width", m_nWidth);
	GetApp()->WriteProfileInt(DISPLAY_SECTION, "Height", m_nHeight);
	GetApp()->WriteProfileInt(DISPLAY_SECTION, "BitsPerPixel", m_nBitsPerPixel);
	GetApp()->WriteProfileInt(DISPLAY_SECTION, "Frequency", m_nFrequency);
}

bool CGameApp::InitInstance()
{
	// Register the window class and create the window
	WNDCLASS wc = {CS_OWNDC | CS_VREDRAW | CS_HREDRAW, (WNDPROC)WindowProc, 0, 0, m_hInstance, LoadIcon(m_hInstance, MAKEINTRESOURCE(IDR_APPLICATION)), LoadCursor((HINSTANCE)NULL, IDC_ARROW), (HBRUSH)GetStockObject(BLACK_BRUSH), MAKEINTRESOURCE(IDR_APPLICATION), m_szAppName};
	if(!RegisterClass(&wc))
	{
		MessageBox("Unable to register window class, aborting.");
		return false;
	}

	// Load the last saved display mode and use it to initialize the window
	LoadDisplaySettings();
	return InitMode(IsFullScreen(), GetWidth(), GetHeight(), GetBitsPerPixel(), GetFrequency());
}

bool CGameApp::InitMode(bool bFullScreen, int nWidth, int nHeight, int nBitsPerPixel, int nFrequency)
{
	if(m_hWnd)
	{
		ResetDisplayMode();
		DestroyWindow();
		m_hWnd = NULL;
	}
	SetFullScreen(bFullScreen);
	SetWidth(nWidth);
	SetHeight(nHeight);
	SetBitsPerPixel(nBitsPerPixel);
	SetFrequency(nFrequency);

	DWORD dwStyle = WS_CLIPCHILDREN | WS_CLIPSIBLINGS | (IsFullScreen() ? WS_POPUP : WS_OVERLAPPEDWINDOW);
#ifdef _DEBUG
	DWORD dwExStyle = 0;
#else
	DWORD dwExStyle = (IsFullScreen() ? WS_EX_TOPMOST : 0);
#endif
	CRect rect(0, 0, GetWidth(), GetHeight());
	if(!CreateEx(m_hInstance, m_szAppName, m_szAppName, dwExStyle, dwStyle, &rect))
	{
		MessageBox("Unable to create application window, aborting.");
		return false;
	}
	if(!IsFullScreen())
	{
		CalcWindowRect(&rect);
		MoveWindow(0, 0, rect.Width(), rect.Height(), false);
	}
	ShowWindow(m_nShowCmd);
	UpdateWindow();
	return true;
}

int CGameApp::ExitInstance()
{
	UnregisterClass(m_szAppName, m_hInstance);
	SaveDisplaySettings();
	ResetDisplayMode();
	return 0;
}

bool CGameApp::OnIdle()
{
	if(!m_bActive)
		return false;
	int nTimer = timeGetTime();
	m_pGameEngine->RenderFrame(nTimer-m_nTimer);
	SwapBuffers(m_hDC);
	m_nTimer = nTimer;
	return true;
}

void CGameApp::Pause()
{
	if(m_bActive)
	{
#ifndef _DEBUG
		SetThreadPriority(GetCurrentThread(), THREAD_PRIORITY_LOWEST);
#endif
		m_pGameEngine->Pause();
		m_bActive = false;
		if(IsFullScreen())
		{
			ShowWindow(SW_MINIMIZE);
			ResetDisplayMode();
		}
	}
}

void CGameApp::Restore()
{
	if(!m_bActive)
	{
#ifndef _DEBUG
		SetThreadPriority(GetCurrentThread(), THREAD_PRIORITY_HIGHEST);
#endif
		m_bActive = true;
		if(IsFullScreen())
		{
			ShowWindow(SW_SHOWNORMAL);
			SetDisplayMode();
		}
		m_nTimer = timeGetTime();
		m_pGameEngine->Restore();
	}
}

int CGameApp::OnCreate(HWND hWnd) 
{
	PIXELFORMATDESCRIPTOR pfdDesc;
	memset((char *)&pfdDesc, 0, sizeof(PIXELFORMATDESCRIPTOR));
	pfdDesc.nSize = sizeof(PIXELFORMATDESCRIPTOR);
	pfdDesc.nVersion = 1;
	pfdDesc.dwFlags = PFD_DRAW_TO_WINDOW | PFD_SUPPORT_OPENGL | PFD_DOUBLEBUFFER;
	pfdDesc.iPixelType = PFD_TYPE_RGBA;
	pfdDesc.iLayerType = PFD_MAIN_PLANE;
	pfdDesc.cColorBits = 32;
	pfdDesc.cAlphaBits = 8;
	pfdDesc.cDepthBits = 32;
	pfdDesc.cStencilBits = 32;

	m_hWnd = hWnd;
	m_hDC = ::GetDC(m_hWnd);
	int nPixelIndex = ChoosePixelFormat(m_hDC, &pfdDesc);
	if(!SetPixelFormat(m_hDC, nPixelIndex, &pfdDesc))
	{
		MessageBox("Error finding a suitable pixel format.");
		return -1;
	}
	DescribePixelFormat(m_hDC, GetPixelFormat(m_hDC), sizeof(PIXELFORMATDESCRIPTOR), &pfdDesc);

	m_hGLRC = wglCreateContext(m_hDC);
	if(!m_hGLRC || !wglMakeCurrent(m_hDC, m_hGLRC))
	{
		MessageBox("Error creating OpenGL rendering context.");
		return -1;
	}

	m_pGameEngine = new CGameEngine;
	return 0;
}

void CGameApp::OnDestroy()
{
	if(m_pGameEngine)
	{
		delete m_pGameEngine;
		m_pGameEngine = NULL;
	}

	if(wglGetCurrentContext())
		wglMakeCurrent(NULL, NULL);
	if(m_hGLRC)
	{
		wglDeleteContext(m_hGLRC);
		m_hGLRC = NULL;
	}
	if(m_hDC)
	{
		::ReleaseDC(m_hWnd, m_hDC);
		m_hDC = NULL;
	}
}

void CGameApp::OnSize(int nType, int nWidth, int nHeight)
{
	if(!nHeight || !nWidth)
		return;
	glViewport(0, 0, nWidth, nHeight);
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	gluPerspective(45.0, (double)nWidth / (double)nHeight, MIN_DISTANCE, MAX_DISTANCE*1.05);
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
	if(!IsFullScreen())
	{
		CRect rect;
		GetClientRect(&rect);
		SetWidth(rect.Width());
		SetHeight(rect.Height());
	}
}

LRESULT CALLBACK CGameApp::WindowProc(HWND hWnd, UINT nMsg, WPARAM wParam, LPARAM lParam)
{
	CRect rect;
	switch(nMsg)
	{
		case WM_CREATE:
			return GameApp()->OnCreate(hWnd);
		case WM_DESTROY:
			GameApp()->OnDestroy();
			break;
		case WM_CLOSE:
			GameApp()->DestroyWindow();
			PostQuitMessage(0);
			return 0;
		case WM_SIZE:
			GameApp()->OnSize(wParam, LOWORD(lParam), HIWORD(lParam));
			break;
		case WM_ACTIVATE:
			if(wParam)
				GameApp()->Restore();
			else
				GameApp()->Pause();
			break;
		case WM_GETMINMAXINFO:
			rect = CRect(0, 0, 320, 240);
			CWnd(hWnd).CalcWindowRect(&rect);
			((MINMAXINFO*)lParam)->ptMinTrackSize.x = rect.Width();
			((MINMAXINFO*)lParam)->ptMinTrackSize.y = rect.Height();
			return 0;
		case WM_SYSCOMMAND:
			// Prevent system commands (like closing, moving, sizing, screensaver, power management, etc) when active
			//if(GameApp()->m_bActive)
			//	return 0;
			break;
		case WM_POWERBROADCAST:
			// Prevent power suspend when active
			if(GameApp()->m_bActive && wParam == PBT_APMQUERYSUSPEND)
				return BROADCAST_QUERY_DENY;
			break;
		case WM_CHAR:
			switch(wParam)
			{
				case 'f':
					GameApp()->InitMode(!GameApp()->IsFullScreen(), 640, 480, 32);
					break;
				case 'p':
					GameApp()->m_pGameEngine->TogglePolygonMode();
					break;
				case 't':
					GameApp()->m_pGameEngine->ToggleTextures();
					break;
				case 'i':
					GameApp()->m_pGameEngine->ToggleInfo();
					break;
				case 'u':
					GameApp()->m_pGameEngine->ToggleUpdate();
					break;
			}
			break;
	}
	return DefWindowProc(hWnd, nMsg, wParam, lParam);
}
