// GLUtil.h : header file
//

#ifndef __GLUtil_h__
#define __GLUtil_h__

#include "glh_genext.h"
#include "PBuffer.h"

// This must be defined before GL_NV_vertex_array_range
class CVertex;
#define VERTEX_ARRAY_SIZE	65535	// Would use 65536, but unsigned short only goes up to 65535
class CVertexArray : public CStackedArray<CVertex, unsigned short>
{
public:
	// Constructor/destructor methods
	CVertexArray() : CStackedArray<CVertex, unsigned short>(VERTEX_ARRAY_SIZE) {}
	void UpdateElement(unsigned short nElement)	{ SetFlags(nElement, Dirty); }
};

class CVertex
{
// Attributes
public:
	static CVertexArray Array;
	CVector m_vPosition;		// The position of the vertex
	CVector m_vNormal;			// The normal of the vertex (used for smooth shading)
	float m_fTexCoord[4];		// The texture coordinates of the vertex

// Operations
public:
	CVertex()						{}
	CVertex(const CVertex &v)		{ *this = v; }
	void operator=(const CVertex &v)
	{
		m_vPosition = v.m_vPosition;
		m_vNormal = v.m_vNormal;
		m_fTexCoord[0] = v.m_fTexCoord[0];
		m_fTexCoord[1] = v.m_fTexCoord[1];
		m_fTexCoord[2] = v.m_fTexCoord[2];
		m_fTexCoord[3] = v.m_fTexCoord[3];
	}
	void Init(CVector &v)
	{
		m_vPosition = v;
		m_vNormal = v;
		m_fTexCoord[0] = m_fTexCoord[1] = 0;
	}
};

class CGLUtil
{
// Attributes
protected:
	// Standard OpenGL members
	HDC m_hDC;
	HGLRC m_hGLRC;

	// Members to indicate whether certain OpenGL extensions are supported
	bool m_bVertexArrayRanges;
	bool m_bCompiledVertexArrays;
	bool m_bPBuffers;
	bool m_bRegisterCombiners;
	int m_nMaxTextureUnits;

	// Members for GL_NV_vertex_array_range and GL_NV_fence extensions
	CVertex *m_pArray;
	GLuint m_nFence;

	// Members for the WGL_ARB_pbuffer extension
	CPBuffer m_PBuffer[8];

public:
	static CGLUtil *m_pMain;

// Operations
public:
	CGLUtil();
	~CGLUtil();
	void Init();
	void Cleanup();
	void InitRenderContext(HDC hDC=NULL, HGLRC hGLRC=NULL);

	HDC GetHDC()					{ return m_hDC; }
	HGLRC GetHGLRC()				{ return m_hGLRC; }

	int GetVertexArraySize()		{ return VERTEX_ARRAY_SIZE * sizeof(CVertex); }
	bool HasVertexArrayRanges()		{ return m_bVertexArrayRanges; }
	bool HasCompiledVertexArrays()	{ return m_bCompiledVertexArrays; }
	bool HasPBuffers()				{ return m_bPBuffers; }
	bool HasRegisterCombiners()		{ return m_bRegisterCombiners; }
	int GetMaxTextureUnits()		{ return m_nMaxTextureUnits; }

	void EnableVertexArray()
	{
		glEnableClientState(GL_VERTEX_ARRAY);
		glVertexPointer(3, GL_FLOAT, sizeof(CVertex), &m_pArray->m_vPosition);
	}
	void DisableVertexArray()
	{
		glDisableClientState(GL_VERTEX_ARRAY);
	}
	void EnableNormalArray()
	{
		glEnableClientState(GL_NORMAL_ARRAY);
		glNormalPointer(GL_FLOAT, sizeof(CVertex), &m_pArray->m_vNormal);
	}
	void DisableNormalArray()
	{
		glDisableClientState(GL_NORMAL_ARRAY);
	}
	void EnableTextureCoordArray(int nTextureUnit=GL_TEXTURE0_ARB, int nOffset=0)
	{
		if(m_nMaxTextureUnits > 1)
			glClientActiveTextureARB(nTextureUnit);
		glEnableClientState(GL_TEXTURE_COORD_ARRAY);
		glTexCoordPointer(2, GL_FLOAT, sizeof(CVertex), &m_pArray->m_fTexCoord[nOffset]);
	}
	void DisableTextureCoordArray(int nTextureUnit=GL_TEXTURE0_ARB)
	{
		if(m_nMaxTextureUnits > 1)
			glClientActiveTextureARB(nTextureUnit);
		glDisableClientState(GL_TEXTURE_COORD_ARRAY);
	}

	void BeginVertexArray();
	void EndVertexArray();

	void BeginPBuffer(int nSize)
	{
		nSize >>= 2;
		for(int nIndex = 7; nSize > 1; nIndex--)
			nSize >>= 1;
		m_PBuffer[nIndex].MakeCurrent();
	}
	void EndPBuffer()				{ wglMakeCurrent(m_hDC, m_hGLRC); }
};

inline CGLUtil *GLUtil()			{ return CGLUtil::m_pMain; }

#endif // __GLUtil_h__
